// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "hw_si.h"

#define SIDEGUB if(g::si_log) DEGUB

void HwSI::ww_out(Hardware *h, WORD offset, DWORD data) {
	//if(data != 0x00400300) {
	SIDEGUB("SI ww_out: 0x%04X, 0x%08X\n", offset, data);
	//throw hardware_fatal_exception("SI Unrecognized command");
	//}
	h->hww(offset, data);
}

void HwSI::ww_exilk(Hardware *, WORD offset, DWORD data) {
	SIDEGUB("SI ww_exilk: 0x%04X, 0x%08X\n", offset, data);
	//this should be enforced somehow

	if(data != 0)
		throw hardware_fatal_exception("SI EXI Lock!");
}

void HwSI::ww_cr(Hardware *h, WORD offset, DWORD data) {
	MYASSERT(offset == SI_CR);
	SIDEGUB("SI ww_cr: 0x%08X\n", data);
	//Clear interrupts if requested
	data = CLEARINTFLAGS(h->hrw(offset), data, SICR_TCINT);
	if(data & SICR_TSTART) {  //Serial transfer
		data = h->si_start(data);	
	}
	data &= ~SICR_RDSTINT;
	h->hww(offset, data);
}
DWORD HwSI::rw_cr(Hardware *h, WORD offset) {
	MYASSERT(offset == SI_CR);
	DWORD data = h->hrw(offset) |
		((h->hrw(SI_SR) & SI_ALLCHANS(SISR_RDST)) ? SICR_RDSTINT : 0);
	SIDEGUB("SI rw_cr: 0x%08X\n", data);
	return data;
}

void HwSI::ww_sr(Hardware *h, WORD offset, DWORD data) {
	MYASSERT(offset == SI_SR);
	DWORD old = h->hrw(offset);
	SIDEGUB("SI ww_sr: 0x%08X + 0x%08X => ", old, data);
	data &= ~(SISR_WR | SI_ALLCHANS(SISR_WRST) | SI_ALLCHANS(SISR_RDST));
	data |= old & SI_ALLCHANS(SISR_RDST);
	data = CLEARINTFLAGS(old, data, SI_ALLCHANS(SISR_NOREP) | SI_ALLCHANS(SISR_COLL) |
		SI_ALLCHANS(SISR_OVRUN) | SI_ALLCHANS(SISR_UNRUN));
	SIDEGUB("0x%08X\n", data);
	h->hww(offset, data);
}
DWORD HwSI::rw_sr(Hardware *h, WORD offset) {
	MYASSERT(offset == SI_SR);
	DWORD data = h->hrw(offset);
	SIDEGUB("SI rw_sr: 0x%08X\n", data);
	return data;
}

DWORD HwSI::rw_in(Hardware *h, WORD offset) {
	//I suppose we don't really need very accurate timing here
	//Except in EXACT_REAL mode. Then we should emulate the SIPOLL register properly.
	static DWORD pt;
	DWORD ct = GetTickCount();
	if((ct - pt) > 1000/60) {
		h->update_pads();
		pt = ct;
	}
	int channel = (offset & 0xFF) / 0xC;
	if((channel*0xC + 0x6404) == offset) {
		ECLEARHWFLAGS(SI_SR, SISR_RDST(channel));
	}
	SIDEGUB("Pad %i read (0x%04X = 0x%08X)\n", channel, offset, h->hrw(offset));
	return h->hrw(offset);
}

void HwSI::ww_poll(Hardware *h, WORD offset, DWORD data) {
	MYASSERT(offset == SI_POLL);
	SIDEGUB("SI ww_poll: 0x%08X\n", data);
	DWORD y = getbitsr(data, 15, 8);
	SIDEGUB("Poll settings: X %i | Y %i | EN/VBCPY bits 0x%02X\n", getbitsr(data, 25, 16),
		y, getbitsr(data, 7, 0));
	if((data & 0xF) != 0/* || y != 2*/) {
		throw hardware_fatal_exception("Unemulated SIPOLL mode!");
	}
	h->hww(offset, data);
}
DWORD HwSI::rw_poll(Hardware *h, WORD offset) {
	MYASSERT(offset == SI_POLL);
	DWORD data = h->hrw(offset);
	SIDEGUB("SI rw_poll: 0x%08X\n", data);
	return data;
}

DWORD Hardware::si_start(DWORD data) {
	int channel = getbitsr(data, 2, 1);
	int out = getbitsr(data, 22, 16);
	if(out == 0) out = 128;
	int in = getbitsr(data, 14, 8);
	if(in == 0) in = 128;
	bool rsi = getbitr(data, 27);
	bool tci = getbitr(data, 30);
	bool callback = getbitr(data, 6);
	bool command = getbitr(data, 7);
	bool channel_enabled = getbitr(data, 24);

	SIDEGUB("Serial transfer: Channel %i, Out %i, In %i,\n"
		"Read Status Interrupt %s, Transfer Complete Interrupt %s,\n"
		"Callback %s, Command %s, Channel %s, ChaNr %i\n",
		channel, out, in, abled(rsi), abled(tci), abled(callback), abled(command),
		abled(channel_enabled), getbitsr(data, 26, 25));

	if(callback || command || channel_enabled)
		throw hardware_fatal_exception("SI Unemulated Transfer flag!");

	//we should disable polling until next vblank
	//we should create proper error codes for disconnected controllers

	bool error = false;
	//if(g::pad_on[channel]) {
	int expected_in;
	BYTE cmd = hrb(0x6480);
	if(out == 1 && cmd == 0x00) { //Get ID
		SIDEGUB("Get ID\n");
		hww(0x6480, g::pad_on[channel] ? 0x09000000 : 0); //standard gc controller
		expected_in = 3;
	} else if(out == 1 && cmd == 0x41) {  //Get origins
		SIDEGUB("Get origins\n");
		/*hwh(0x6480, 0x4100);  //Dolwin style
		hww(0x6480 + 2, 0x80808080);
		hwh(0x6480 + 6, 0x1F);*/
		hwb(0x6480 + 1, 0x00);  //Gcube style
		hww(0x6480 + 2, 0x80808080);
		hww(0x6480 + 6, 0x1F1F1F1F);
		expected_in = (in == 9 || in == 10) ? in : 9;
	} else if(out == 3 && cmd == 0x42) {  //Calibrate?
		hwb(0x6480 + 1, 0x00);  //Gcube style
		hww(0x6480 + 2, 0x80808080);
		hww(0x6480 + 6, 0x1F1F1F1F);
		expected_in = 10;
	} else {
		SIDEGUB("SI Unknown command: 0x%02X\n", cmd);
		//if(g::bouehr)
		throw bouehr_exception("SI Unknown serial command");
		//error = true;
	}
	if(!error && in != expected_in) {
		SIDEGUB("SI Unexpected in length: %i | expected: %i\n", in, expected_in);
		//if(g::bouehr)
		throw bouehr_exception("SI Unexpected in length");
	}
	/*} else {
	DEGUB("SI Unconnected Pad\n");
	error = true;
	}*/

	/*if(error) {
	DEGUB("SI Error set\n");
	SETHWFLAGS(SI_SR, SISR_NOREP(channel));// | SISR_RDST(channel));
	data |= SICR_COMERR;
	} else*/ {
		CLEARHWFLAGS(SI_SR, SISR_NOREP(channel));
		data &= ~SICR_COMERR;
	}
	{
		SIDEGUB("SI Transfer complete\n");
		data &= ~SICR_TSTART;

		// set completion interrupt
		data |= SICR_TCINT;
		// generate cpu interrupt (if enabled)
		if(data & SICR_TCINTMSK) {
			interrupt.raise(INTEX_SI, "SI Transfer complete");
		}
	}
	return data;
}

//unused for now
/*#define PADA_MIDPOINT 128
#define PADA_POWER_UNCLAMPED 128
#define PADA_POWER_CLAMPED 90
#define PADA_POWER\
(g::pad_settings[padn].clamped ? PADA_POWER_CLAMPED : PADA_POWER_UNCLAMPED)
#define PADA_DEADZONE 15*/

BYTE analog_power(bool half, bool quarter) {
	if(quarter)
		return 32;
	else if(half)
		return 64;
	else
		return 127;
}

#define PADD_LEFT 0x0001
#define PADD_RIGHT 0x0002
#define PADD_DOWN 0x0004
#define PADD_UP 0x0008
#define PADD_Z 0x0010
#define PADD_R 0x0020
#define PADD_L 0x0040
//#define PADD_UNDEFINED 0x0080
#define PADD_A 0x0100
#define PADD_B 0x0200
#define PADD_X 0x0400
#define PADD_Y 0x0800
#define PADD_START 0x1000
//#define PADD_UNDEFINED 0x2000
//#define PADD_UNDEFINED 0x4000
//#define PADD_UNDEFINED 0x8000

void Hardware::update_pads() {
	if(g::beep) {
		GLE(Beep(200, 100));
	}

#define PRESSED(key) ((buffer[key] & 0x80) && m_window_active)
	BYTE buffer[256];
	HWHR(m_pdiDevice->GetDeviceState(sizeof(buffer), (LPVOID)&buffer));
	//PrintPacket(buffer, 256);

	//is this buffered or immediate, ie. should I clear the flag if the button isn't pressed?
	if(PRESSED(g::reset_button)) {
		SETHWFLAGS(0x3000, 0x00010000);
	}

	bool half = PRESSED(g::pad_half) != 0;
	bool quarter = PRESSED(g::pad_quarter) != 0;

#define DPADMAP(macro) \
	macro(a, PADD_A)\
	macro(b, PADD_B)\
	macro(x, PADD_X)\
	macro(y, PADD_Y)\
	macro(z, PADD_Z)\
	macro(start, PADD_START)\
	macro(dup, PADD_UP)\
	macro(dleft, PADD_LEFT)\
	macro(ddown, PADD_DOWN)\
	macro(dright, PADD_RIGHT)\

#define DO_DPAD(id, flag) if(PRESSED(g::pad_sc[i].id)) pd |= flag;

	for(WORD i=0; i<PAD_NPADS; i++) if(g::pad_on[i]) {
		WORD pd=0;

		DPADMAP(DO_DPAD);
		if(PRESSED(g::pad_sc[i].l) && !half && !quarter)
			pd |= PADD_L;
		if(PRESSED(g::pad_sc[i].r) && !half && !quarter)
			pd |= PADD_R;

		hwh(0x6404 + i*0xC, pd);// | PADD_A); //force-press A button for degub
		//DEGUB("Around cycle %i\n", prevCycles);

		//Analog stick
		hwb(0x6406 + i*0xC,
			128 + (PRESSED(g::pad_sc[i].aright) ? analog_power(half, quarter) : 0) -
			(PRESSED(g::pad_sc[i].aleft) ? analog_power(half, quarter) : 0));
		hwb(0x6407 + i*0xC,
			128 + (PRESSED(g::pad_sc[i].aup) ? analog_power(half, quarter) : 0) -
			(PRESSED(g::pad_sc[i].adown) ? analog_power(half, quarter) : 0));
		//C-stick
		hwb(0x6408 + i*0xC,
			128 + (PRESSED(g::pad_sc[i].cright) ? analog_power(half, quarter) : 0) -
			(PRESSED(g::pad_sc[i].cleft) ? analog_power(half, quarter) : 0));
		hwb(0x6409 + i*0xC,
			128 + (PRESSED(g::pad_sc[i].cup) ? analog_power(half, quarter) : 0) -
			(PRESSED(g::pad_sc[i].cdown) ? analog_power(half, quarter) : 0));
		//Analog L/R  //something wrong here?
		hwb(0x640A + i*0xC,
			(PRESSED(g::pad_sc[i].l) ? analog_power(half, quarter) + 0x1F : 0));
		hwb(0x640B + i*0xC,
			(PRESSED(g::pad_sc[i].r) ? analog_power(half, quarter) + 0x1F : 0));
	}
}
